// API URL - update this to your server
// const API_URL = "http://localhost:8080";
const API_URL = "https://kafica.dimitrije.online";

// Transform coffee type to emoji
function getCoffeeEmoji(coffeeType) {
  const emojiMap = {
    Black: "☕",
    NES: "🥤",
    IDK: "🤷",
  };
  return emojiMap[coffeeType] || "☕";
}

document.addEventListener("DOMContentLoaded", async () => {
  // Get all DOM elements
  const registerSection = document.getElementById("registerSection");
  const apiKeySection = document.getElementById("apiKeySection");
  const resendSection = document.getElementById("resendSection");
  const mainSection = document.getElementById("mainSection");

  const regEmailInput = document.getElementById("regEmail");
  const regNameInput = document.getElementById("regName");
  const registerBtn = document.getElementById("registerBtn");

  const apiKeyInput = document.getElementById("apiKey");
  const saveApiKeyBtn = document.getElementById("saveApiKey");
  const showRegisterSectionBtn = document.getElementById("showRegisterSection");
  const showRegisterFromLoginBtn = document.getElementById(
    "showRegisterFromLogin"
  );
  const showResendSectionBtn = document.getElementById("showResendSection");
  const backToLoginBtn = document.getElementById("backToLogin");

  const resendEmailInput = document.getElementById("resendEmail");
  const resendActivationBtn = document.getElementById("resendActivationBtn");

  const userNameSpan = document.getElementById("userName");
  const logoutBtn = document.getElementById("logout");

  const createSessionBtn = document.getElementById("createSessionBtn");
  const joinSessionBtn = document.getElementById("joinSessionBtn");
  const leaveSessionBtn = document.getElementById("leaveSessionBtn");

  const sessionStatus = document.getElementById("sessionStatus");
  const status = document.getElementById("status");

  // Version notification elements
  const versionNotification = document.getElementById("versionNotification");
  const versionMessage = document.getElementById("versionMessage");
  const dismissVersionBtn = document.getElementById("dismissVersion");

  // Group elements
  const groupSelect = document.getElementById("groupSelect");
  const createGroupBtn = document.getElementById("createGroupBtn");
  const joinGroupBtn = document.getElementById("joinGroupBtn");
  const getCodeBtn = document.getElementById("getCodeBtn");
  const viewMembersBtn = document.getElementById("viewMembersBtn");
  const leaveGroupBtn = document.getElementById("leaveGroupBtn");
  const createGroupModal = document.getElementById("createGroupModal");
  const joinGroupModal = document.getElementById("joinGroupModal");
  const inviteCodeModal = document.getElementById("inviteCodeModal");
  const membersModal = document.getElementById("membersModal");
  const newGroupNameInput = document.getElementById("newGroupName");
  const joinGroupIdInput = document.getElementById("joinGroupId");
  const inviteCodeDisplay = document.getElementById("inviteCodeDisplay");
  const membersList = document.getElementById("membersList");
  const confirmCreateGroup = document.getElementById("confirmCreateGroup");
  const cancelCreateGroup = document.getElementById("cancelCreateGroup");
  const confirmJoinGroup = document.getElementById("confirmJoinGroup");
  const cancelJoinGroup = document.getElementById("cancelJoinGroup");
  const copyInviteCode = document.getElementById("copyInviteCode");
  const closeInviteCode = document.getElementById("closeInviteCode");
  const closeMembersModal = document.getElementById("closeMembersModal");

  // Change name elements
  const changeNameBtn = document.getElementById("changeNameBtn");
  const changeNameModal = document.getElementById("changeNameModal");
  const newDisplayName = document.getElementById("newDisplayName");
  const confirmChangeName = document.getElementById("confirmChangeName");
  const cancelChangeName = document.getElementById("cancelChangeName");

  let currentGroups = [];

  // Load saved data and determine which section to show
  const data = await chrome.storage.local.get([
    "apiKey",
    "userName",
    "preferredCoffeeType",
  ]);

  if (data.apiKey && data.userName) {
    showMainSection(data.userName);
    // Restore preferred coffee type
    if (data.preferredCoffeeType) {
      const coffeeRadio = document.querySelector(
        `input[name="coffeeType"][value="${data.preferredCoffeeType}"]`
      );
      if (coffeeRadio) {
        coffeeRadio.checked = true;
      }
    }
  } else {
    await showApiKeySection();
  }

  // ===== Section Management =====

  function showRegisterSection() {
    registerSection.style.display = "block";
    apiKeySection.style.display = "none";
    resendSection.style.display = "none";
    mainSection.style.display = "none";
  }

  async function showApiKeySection() {
    registerSection.style.display = "none";
    apiKeySection.style.display = "block";
    resendSection.style.display = "none";
    mainSection.style.display = "none";
  }

  function showResendSection() {
    registerSection.style.display = "none";
    apiKeySection.style.display = "none";
    resendSection.style.display = "block";
    mainSection.style.display = "none";
  }

  async function showMainSection(userName) {
    registerSection.style.display = "none";
    apiKeySection.style.display = "none";
    resendSection.style.display = "none";
    mainSection.style.display = "block";
    userNameSpan.textContent = userName;
    await loadGroups();
    loadSession();
    checkVersion();
  }

  // ===== Version Checking =====

  function compareVersions(v1, v2) {
    const parts1 = v1.split(".").map(Number);
    const parts2 = v2.split(".").map(Number);

    for (let i = 0; i < Math.max(parts1.length, parts2.length); i++) {
      const part1 = parts1[i] || 0;
      const part2 = parts2[i] || 0;

      if (part1 < part2) return -1;
      if (part1 > part2) return 1;
    }
    return 0;
  }

  async function checkVersion() {
    try {
      const manifest = chrome.runtime.getManifest();
      const currentVersion = manifest.version;

      const dismissedData = await chrome.storage.local.get([
        "dismissedVersion",
      ]);
      if (dismissedData.dismissedVersion === currentVersion) {
        return;
      }

      const response = await fetch(`${API_URL}/api/version`);
      if (response.ok) {
        const data = await response.json();
        const latestVersion = data.latest_version;

        if (compareVersions(currentVersion, latestVersion) < 0) {
          versionMessage.textContent = `New version ${latestVersion} available! Current: ${currentVersion}`;
          versionNotification.style.display = "flex";
        }
      }
    } catch (err) {
      console.error("Failed to check version:", err);
    }
  }

  dismissVersionBtn.addEventListener("click", async () => {
    versionNotification.style.display = "none";
    const manifest = chrome.runtime.getManifest();
    await chrome.storage.local.set({ dismissedVersion: manifest.version });
  });

  // ===== Groups =====

  async function loadGroups() {
    const data = await chrome.storage.local.get(["apiKey", "selectedGroupId"]);
    if (!data.apiKey) return;

    try {
      const response = await fetch(`${API_URL}/api/groups/list`, {
        headers: { Authorization: `Bearer ${data.apiKey}` },
      });

      if (response.ok) {
        const result = await response.json();
        const groups = result.groups || [];

        currentGroups = groups;

        if (groups.length === 0) {
          groupSelect.innerHTML =
            '<option value="">No groups - create or join one</option>';
        } else {
          groupSelect.innerHTML =
            '<option value="">-- Select a group --</option>';
          groups.forEach((group) => {
            const option = document.createElement("option");
            option.value = group.id;
            option.textContent = `${group.name} (${group.members.length} members)`;
            groupSelect.appendChild(option);
          });
        }

        if (
          data.selectedGroupId &&
          groups.some((g) => g.id === data.selectedGroupId)
        ) {
          groupSelect.value = data.selectedGroupId;
        } else {
          await chrome.storage.local.set({ selectedGroupId: "" });
        }

        updateGroupButtons();
      }
    } catch (err) {
      console.error("Failed to load groups:", err);
    }
  }

  function updateGroupButtons() {
    const selectedGroupId = groupSelect.value;
    const hasSelection = selectedGroupId !== "";
    leaveGroupBtn.style.display = hasSelection ? "block" : "none";
    getCodeBtn.style.display = hasSelection ? "block" : "none";
    viewMembersBtn.style.display = hasSelection ? "block" : "none";
  }

  groupSelect.addEventListener("change", async () => {
    const groupId = groupSelect.value;
    await chrome.storage.local.set({ selectedGroupId: groupId });
    updateGroupButtons();
    loadSession();
    chrome.runtime.sendMessage({ action: "refresh" });
  });

  createGroupBtn.addEventListener("click", () => {
    createGroupModal.style.display = "flex";
    newGroupNameInput.value = "";
    newGroupNameInput.focus();
  });

  cancelCreateGroup.addEventListener("click", () => {
    createGroupModal.style.display = "none";
  });

  confirmCreateGroup.addEventListener("click", async () => {
    const name = newGroupNameInput.value.trim();
    if (!name) {
      showStatus("Please enter a group name", "error");
      return;
    }

    const data = await chrome.storage.local.get(["apiKey"]);
    if (!data.apiKey) return;

    try {
      const response = await fetch(`${API_URL}/api/groups/create`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${data.apiKey}`,
        },
        body: JSON.stringify({ name }),
      });

      if (response.ok) {
        const result = await response.json();
        createGroupModal.style.display = "none";

        await navigator.clipboard.writeText(result.group_id);
        showStatus(`Group created! ID copied to clipboard`, "success");

        await loadGroups();
        groupSelect.value = result.group_id;
        await chrome.storage.local.set({ selectedGroupId: result.group_id });
        updateGroupButtons();
        loadSession();
      } else {
        const text = await response.text();
        showStatus(text || "Failed to create group", "error");
      }
    } catch (err) {
      showStatus("Connection error: " + err.message, "error");
    }
  });

  joinGroupBtn.addEventListener("click", () => {
    joinGroupModal.style.display = "flex";
    joinGroupIdInput.value = "";
    joinGroupIdInput.focus();
  });

  cancelJoinGroup.addEventListener("click", () => {
    joinGroupModal.style.display = "none";
  });

  confirmJoinGroup.addEventListener("click", async () => {
    const groupId = joinGroupIdInput.value.trim();
    if (!groupId) {
      showStatus("Please enter a group ID", "error");
      return;
    }

    const data = await chrome.storage.local.get(["apiKey"]);
    if (!data.apiKey) return;

    try {
      const response = await fetch(`${API_URL}/api/groups/join`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${data.apiKey}`,
        },
        body: JSON.stringify({ group_id: groupId }),
      });

      if (response.ok) {
        const result = await response.json();
        joinGroupModal.style.display = "none";
        showStatus(`Joined "${result.group.name}"!`, "success");
        await loadGroups();
        groupSelect.value = groupId;
        await chrome.storage.local.set({ selectedGroupId: groupId });
        updateGroupButtons();
        loadSession();
      } else {
        const text = await response.text();
        showStatus(text || "Failed to join group", "error");
      }
    } catch (err) {
      showStatus("Connection error: " + err.message, "error");
    }
  });

  leaveGroupBtn.addEventListener("click", async () => {
    const groupId = groupSelect.value;
    if (!groupId) return;

    if (!confirm("Are you sure you want to leave this group?")) {
      return;
    }

    const data = await chrome.storage.local.get(["apiKey"]);
    if (!data.apiKey) return;

    try {
      const response = await fetch(`${API_URL}/api/groups/leave`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${data.apiKey}`,
        },
        body: JSON.stringify({ group_id: groupId }),
      });

      if (response.ok) {
        showStatus("Left group", "success");
        groupSelect.value = "";
        await chrome.storage.local.set({ selectedGroupId: "" });
        await loadGroups();
        updateGroupButtons();
        loadSession();
      } else {
        const text = await response.text();
        showStatus(text || "Failed to leave group", "error");
      }
    } catch (err) {
      showStatus("Connection error: " + err.message, "error");
    }
  });

  getCodeBtn.addEventListener("click", async () => {
    const groupId = groupSelect.value;
    if (!groupId) return;

    const data = await chrome.storage.local.get(["apiKey"]);
    if (!data.apiKey) return;

    try {
      const response = await fetch(
        `${API_URL}/api/groups/code?group_id=${encodeURIComponent(groupId)}`,
        {
          headers: { Authorization: `Bearer ${data.apiKey}` },
        }
      );

      if (response.ok) {
        const result = await response.json();
        inviteCodeDisplay.value = result.invitation_code;
        inviteCodeModal.style.display = "flex";
      } else {
        const text = await response.text();
        showStatus(text || "Failed to get invitation code", "error");
      }
    } catch (err) {
      showStatus("Connection error: " + err.message, "error");
    }
  });

  copyInviteCode.addEventListener("click", async () => {
    try {
      await navigator.clipboard.writeText(inviteCodeDisplay.value);
      showStatus("Invitation code copied to clipboard!", "success");
      inviteCodeModal.style.display = "none";
    } catch (err) {
      showStatus("Failed to copy to clipboard", "error");
    }
  });

  closeInviteCode.addEventListener("click", () => {
    inviteCodeModal.style.display = "none";
  });

  viewMembersBtn.addEventListener("click", () => {
    const selectedGroupId = groupSelect.value;
    if (!selectedGroupId) return;

    const group = currentGroups.find((g) => g.id === selectedGroupId);
    if (!group) return;

    if (group.members && group.members.length > 0) {
      membersList.innerHTML = group.members
        .map(
          (email) => `
        <div class="member-item">${email}</div>
      `
        )
        .join("");
    } else {
      membersList.innerHTML =
        '<div class="no-members">No members in this group</div>';
    }

    membersModal.style.display = "flex";
  });

  closeMembersModal.addEventListener("click", () => {
    membersModal.style.display = "none";
  });

  // ===== Auth =====

  registerBtn.addEventListener("click", async () => {
    const email = regEmailInput.value.trim();
    const name = regNameInput.value.trim();

    if (!email || !name) {
      showStatus("Please fill in all fields", "error");
      return;
    }

    try {
      const response = await fetch(`${API_URL}/api/register`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ email, name }),
      });

      if (response.ok) {
        await chrome.storage.local.set({ userName: name });
        showStatus("Check your email for verification!", "success");
        await showApiKeySection();
      } else {
        const text = await response.text();
        showStatus(text || "Registration failed", "error");
      }
    } catch (err) {
      showStatus("Connection error: " + err.message, "error");
    }
  });

  showRegisterFromLoginBtn.addEventListener("click", () => {
    showRegisterSection();
  });

  showRegisterSectionBtn.addEventListener("click", async () => {
    await showApiKeySection();
  });

  showResendSectionBtn.addEventListener("click", () => {
    showResendSection();
  });

  backToLoginBtn.addEventListener("click", async () => {
    await showApiKeySection();
  });

  saveApiKeyBtn.addEventListener("click", async () => {
    const apiKey = apiKeyInput.value.trim();

    if (!apiKey) {
      showStatus("Please enter your API key", "error");
      return;
    }

    try {
      const response = await fetch(`${API_URL}/api/user`, {
        method: "GET",
        headers: { Authorization: `Bearer ${apiKey}` },
      });

      if (!response.ok) {
        if (response.status === 401) {
          showStatus("Invalid or unverified API key", "error");
        } else {
          showStatus("Failed to verify API key", "error");
        }
        return;
      }

      const userData = await response.json();
      const userName = userData.name;

      await chrome.storage.local.set({ apiKey, userName });
      showStatus("Logged in!", "success");
      showMainSection(userName);
    } catch (err) {
      console.error("Login error:", err);
      showStatus("Failed to connect to server", "error");
    }
  });

  resendActivationBtn.addEventListener("click", async () => {
    const email = resendEmailInput.value.trim();

    if (!email) {
      showStatus("Please enter your email", "error");
      return;
    }

    try {
      const response = await fetch(`${API_URL}/api/resend-activation`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ email }),
      });

      if (response.ok) {
        showStatus("Activation email sent! Check your inbox.", "success");
        resendEmailInput.value = "";
        setTimeout(async () => {
          await showApiKeySection();
        }, 2000);
      } else {
        const text = await response.text();
        showStatus(text || "Failed to resend activation email", "error");
      }
    } catch (err) {
      showStatus("Connection error: " + err.message, "error");
    }
  });

  logoutBtn.addEventListener("click", async () => {
    await chrome.storage.local.clear();
    showStatus("Logged out", "success");
    showRegisterSection();
  });

  changeNameBtn.addEventListener("click", async () => {
    const data = await chrome.storage.local.get(["userName"]);
    newDisplayName.value = data.userName || "";
    changeNameModal.style.display = "flex";
    newDisplayName.focus();
  });

  confirmChangeName.addEventListener("click", async () => {
    const name = newDisplayName.value.trim();

    if (!name) {
      showStatus("Display name cannot be empty", "error");
      return;
    }

    const data = await chrome.storage.local.get(["apiKey"]);
    if (!data.apiKey) {
      showStatus("Please login first", "error");
      return;
    }

    try {
      const response = await fetch(`${API_URL}/api/user/name`, {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${data.apiKey}`,
        },
        body: JSON.stringify({ name }),
      });

      if (response.ok) {
        await chrome.storage.local.set({ userName: name });
        userNameSpan.textContent = name;
        changeNameModal.style.display = "none";
        showStatus("Display name updated!", "success");
      } else {
        const text = await response.text();
        showStatus(text || "Failed to update name", "error");
      }
    } catch (err) {
      console.error("Update name error:", err);
      showStatus("Failed to connect to server", "error");
    }
  });

  cancelChangeName.addEventListener("click", () => {
    changeNameModal.style.display = "none";
  });

  // ===== Session Actions =====

  // Save coffee type preference when changed
  document.querySelectorAll('input[name="coffeeType"]').forEach((radio) => {
    radio.addEventListener("change", async (e) => {
      await chrome.storage.local.set({ preferredCoffeeType: e.target.value });
    });
  });

  function getSelectedCoffeeType() {
    const selected = document.querySelector('input[name="coffeeType"]:checked');
    return selected ? selected.value : "Black";
  }

  createSessionBtn.addEventListener("click", async () => {
    await createSession();
  });

  joinSessionBtn.addEventListener("click", async () => {
    await joinSession();
  });

  leaveSessionBtn.addEventListener("click", async () => {
    await leaveSession();
  });

  async function createSession() {
    const data = await chrome.storage.local.get(["apiKey", "selectedGroupId"]);

    if (!data.apiKey) {
      showStatus("Please login first", "error");
      return;
    }

    const groupId = data.selectedGroupId || "";

    if (!groupId) {
      showStatus("Please select or create a group first", "error");
      return;
    }

    const coffeeType = getSelectedCoffeeType();

    try {
      const response = await fetch(`${API_URL}/api/session/create`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${data.apiKey}`,
        },
        body: JSON.stringify({
          coffee_type: coffeeType,
          group_id: groupId,
        }),
      });

      if (response.ok) {
        showStatus(`Created coffee session with ${coffeeType}!`, "success");
        loadSession();
        chrome.runtime.sendMessage({ action: "refresh" });
      } else {
        const result = await response.json();
        showStatus(result.error || "Failed to create session", "error");
      }
    } catch (err) {
      showStatus("Connection error: " + err.message, "error");
    }
  }

  async function joinSession() {
    const data = await chrome.storage.local.get(["apiKey", "selectedGroupId"]);

    if (!data.apiKey) {
      showStatus("Please login first", "error");
      return;
    }

    const groupId = data.selectedGroupId || "";

    if (!groupId) {
      showStatus("Please select a group first", "error");
      return;
    }

    const coffeeType = getSelectedCoffeeType();

    try {
      const response = await fetch(`${API_URL}/api/session/join`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${data.apiKey}`,
        },
        body: JSON.stringify({
          coffee_type: coffeeType,
          group_id: groupId,
        }),
      });

      if (response.ok) {
        showStatus(`Joined session with ${coffeeType}!`, "success");
        loadSession();
        chrome.runtime.sendMessage({ action: "refresh" });
      } else {
        const result = await response.json();
        showStatus(result.error || "Failed to join session", "error");
      }
    } catch (err) {
      showStatus("Connection error: " + err.message, "error");
    }
  }

  async function leaveSession() {
    const data = await chrome.storage.local.get(["apiKey", "selectedGroupId"]);

    if (!data.apiKey) {
      showStatus("Please login first", "error");
      return;
    }

    const groupId = data.selectedGroupId || "";

    if (!groupId) {
      showStatus("Please select a group first", "error");
      return;
    }

    try {
      const response = await fetch(`${API_URL}/api/session/leave`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${data.apiKey}`,
        },
        body: JSON.stringify({
          group_id: groupId,
        }),
      });

      if (response.ok) {
        showStatus("Left the session", "success");
        loadSession();
        chrome.runtime.sendMessage({ action: "refresh" });
      } else {
        const result = await response.json();
        showStatus(result.error || "Failed to leave session", "error");
      }
    } catch (err) {
      showStatus("Connection error: " + err.message, "error");
    }
  }

  function showStatus(message, type) {
    status.textContent = message;
    status.className = type;
    setTimeout(() => {
      status.textContent = "";
      status.className = "";
    }, 3000);
  }

  async function loadSession() {
    try {
      const storageData = await chrome.storage.local.get([
        "selectedGroupId",
        "userName",
      ]);
      const groupId = storageData.selectedGroupId || "";
      const myName = storageData.userName || "";

      if (!groupId) {
        sessionStatus.innerHTML =
          '<div class="empty">Please select a group</div>';
        createSessionBtn.style.display = "none";
        joinSessionBtn.style.display = "none";
        leaveSessionBtn.style.display = "none";
        return;
      }

      const url = `${API_URL}/api/session?group_id=${encodeURIComponent(
        groupId
      )}`;
      const response = await fetch(url);
      const data = await response.json();

      const activeSession = data.active_session;
      const lastSession = data.last_session;

      if (activeSession) {
        // There's an active session
        const participants = activeSession.participants || [];
        const iAmInSession = participants.some((p) => p.name === myName);

        // Update button visibility
        if (iAmInSession) {
          createSessionBtn.style.display = "none";
          joinSessionBtn.style.display = "none";
          leaveSessionBtn.style.display = "block";
        } else {
          createSessionBtn.style.display = "none";
          joinSessionBtn.style.display = "block";
          leaveSessionBtn.style.display = "none";
        }

        // Calculate time remaining
        const expiresAt = new Date(activeSession.expires_at);
        const now = new Date();
        const minutesLeft = Math.max(0, Math.round((expiresAt - now) / 60000));

        // Build participant list
        let participantHTML = participants
          .map((p) => {
            const creatorBadge = p.is_creator
              ? '<span class="creator-badge">👑</span>'
              : "";
            return `<div class="participant">${creatorBadge}${
              p.name
            } - ${getCoffeeEmoji(p.coffee_type)}</div>`;
          })
          .join("");

        sessionStatus.innerHTML = `
          <div class="session-header">
            <h3>☕ Active Coffee Session</h3>
            <span class="time-remaining">${minutesLeft} min left</span>
          </div>
          <div class="participants">
            ${participantHTML}
          </div>
        `;
      } else if (lastSession) {
        // No active session, show last session
        createSessionBtn.style.display = "block";
        joinSessionBtn.style.display = "none";
        leaveSessionBtn.style.display = "none";

        const participants = lastSession.participants || [];
        const endedAt = new Date(lastSession.ended_at);
        const timeAgo = getTimeAgo(endedAt);

        let participantHTML = participants
          .map((p) => {
            const creatorBadge = p.is_creator
              ? '<span class="creator-badge">👑</span>'
              : "";
            return `<div class="participant">${creatorBadge}${
              p.name
            } - ${getCoffeeEmoji(p.coffee_type)}</div>`;
          })
          .join("");

        sessionStatus.innerHTML = `
          <div class="session-header">
            <h3>Last Coffee Session</h3>
            <span class="time-ago">${timeAgo}</span>
          </div>
          <div class="participants">
            ${participantHTML}
          </div>
        `;
      } else {
        // No sessions at all
        createSessionBtn.style.display = "block";
        joinSessionBtn.style.display = "none";
        leaveSessionBtn.style.display = "none";

        sessionStatus.innerHTML =
          '<div class="empty">No coffee sessions yet. Create one!</div>';
      }
    } catch (err) {
      console.error("Failed to load session:", err);
      sessionStatus.innerHTML =
        '<div class="error">Could not load session status</div>';
      createSessionBtn.style.display = "none";
      joinSessionBtn.style.display = "none";
      leaveSessionBtn.style.display = "none";
    }
  }

  function getTimeAgo(date) {
    const now = new Date();
    const diffMs = now - date;
    const diffMins = Math.round(diffMs / 60000);

    if (diffMins < 1) return "just now";
    if (diffMins === 1) return "1 min ago";
    if (diffMins < 60) return `${diffMins} mins ago`;

    const diffHours = Math.round(diffMins / 60);
    if (diffHours === 1) return "1 hour ago";
    if (diffHours < 24) return `${diffHours} hours ago`;

    const diffDays = Math.round(diffHours / 24);
    if (diffDays === 1) return "1 day ago";
    return `${diffDays} days ago`;
  }
});
