// API URL - update this to your server
// const API_URL = "http://localhost:8080";
const API_URL = "https://kafica.dimitrije.online";

// Initialize alarm on install
chrome.runtime.onInstalled.addListener(() => {
  chrome.alarms.create("checkActive", { periodInMinutes: 1 });
  checkActiveSession();
});

// Also check on startup
chrome.runtime.onStartup.addListener(() => {
  chrome.alarms.create("checkActive", { periodInMinutes: 1 });
  checkActiveSession();
});

// Handle alarm
chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === "checkActive") {
    checkActiveSession();
  }
});

async function checkActiveSession() {
  try {
    const data = await chrome.storage.local.get([
      "userName",
      "selectedGroupId",
      "notifiedSessions",
    ]);
    const myName = data.userName || "";
    const groupId = data.selectedGroupId || "";
    const notifiedSessions = data.notifiedSessions || {};

    if (!groupId) {
      chrome.action.setBadgeText({ text: "" });
      return;
    }

    const url = `${API_URL}/api/session?group_id=${encodeURIComponent(
      groupId
    )}`;
    const response = await fetch(url);
    const result = await response.json();

    const activeSession = result.active_session;

    if (activeSession) {
      const participants = activeSession.participants || [];
      const participantCount = participants.length;

      // Update badge with participant count
      chrome.action.setBadgeText({ text: participantCount.toString() });
      chrome.action.setBadgeBackgroundColor({ color: "#6F4E37" }); // Coffee brown

      // Create a unique session ID based on session_id
      const sessionId = `session:${activeSession.session_id}`;

      // Check if we've already notified about this session
      if (!notifiedSessions[sessionId]) {
        // Find the creator
        const creator = participants.find((p) => p.is_creator);

        if (creator && creator.name !== myName) {
          // Notify about new session
          chrome.notifications.create({
            type: "basic",
            iconUrl: "icon128.png",
            title: "Kafica - Coffee Session Started!",
            message: `${creator.name} started a coffee session with ${creator.coffee_type}!`,
            priority: 1,
          });

          notifiedSessions[sessionId] = Date.now();
        }
      }

      // Notify when someone joins (if not creator)
      for (const participant of participants) {
        const joinId = `join:${sessionId}:${participant.name}`;

        if (
          !participant.is_creator &&
          !notifiedSessions[joinId] &&
          participant.name !== myName
        ) {
          chrome.notifications.create({
            type: "basic",
            iconUrl: "icon128.png",
            title: "Kafica - Someone Joined!",
            message: `${participant.name} joined with ${participant.coffee_type}!`,
            priority: 0,
          });

          notifiedSessions[joinId] = Date.now();
        }
      }
    } else {
      // No active session
      chrome.action.setBadgeText({ text: "" });
    }

    // Clean up old notifications (older than 24 hours)
    const oneDayAgo = Date.now() - 24 * 60 * 60 * 1000;
    for (const key in notifiedSessions) {
      if (notifiedSessions[key] < oneDayAgo) {
        delete notifiedSessions[key];
      }
    }

    // Save updated notified sessions
    await chrome.storage.local.set({ notifiedSessions });
  } catch (err) {
    console.error("Failed to check session status:", err);
    chrome.action.setBadgeText({ text: "?" });
    chrome.action.setBadgeBackgroundColor({ color: "#999" });
  }
}

// Allow manual refresh from popup
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === "refresh") {
    checkActiveSession();
    sendResponse({ success: true });
  }
});
